#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
MaestroDMX Show Group Propagation Tool
-------------------------------------
Purpose:
- Load a source show JSON
- Select a cue + source group (1..4)
- Select a target group (1..4)
- Select a folder with target show JSON files
- Apply: Replace the entire target group block in ALL cues of the selected target show files
- Optionally create timestamped backups

This tool is intentionally "simple": it does not attempt to merge fields; it deep-copies
the full group block from the source cue/group and overwrites the target group block.

Tested against show structure where cues are stored at: root["show"]["patternCue"] (list of cues)
and group blocks are stored under the following keys per cue:
  1 -> "params"
  2 -> "secondaryParams"
  3 -> "tertiaryParams"
  4 -> "quaternaryParams"

The tool also tries to robustly find cue lists in other plausible structures by scanning.
"""

from __future__ import annotations

import copy
import datetime as _dt
import json
import os
import sys
import traceback
from dataclasses import dataclass
from pathlib import Path
from typing import Any, Dict, List, Optional, Tuple

import tkinter as tk
from tkinter import ttk, filedialog, messagebox
import webbrowser


GROUP_KEY_BY_NUM = {
    1: "params",
    2: "secondaryParams",
    3: "tertiaryParams",
    4: "quaternaryParams",
}


def now_ts() -> str:
    return _dt.datetime.now().strftime("%Y%m%d-%H%M%S")


def read_json(path: Path) -> Any:
    with path.open("r", encoding="utf-8") as f:
        return json.load(f)


def write_json(path: Path, obj: Any) -> None:
    # Keep output stable & human-readable. Maestro should accept it.
    with path.open("w", encoding="utf-8") as f:
        json.dump(obj, f, ensure_ascii=False, indent=2, sort_keys=False)
        f.write("\n")


def is_cue_like(obj: Any) -> bool:
    if not isinstance(obj, dict):
        return False
    # Must have at least one group key.
    return any(k in obj for k in GROUP_KEY_BY_NUM.values())


def find_cue_list(doc: Any) -> Optional[Tuple[List[Dict[str, Any]], str]]:
    """
    Returns (cue_list, path_string) or None.
    Prefers common Maestro path: doc["show"]["patternCue"].
    Otherwise scans doc for a list of dicts that look like cues.
    """
    try:
        if isinstance(doc, dict):
            show = doc.get("show")
            if isinstance(show, dict):
                pc = show.get("patternCue")
                if isinstance(pc, list) and pc and all(isinstance(x, dict) for x in pc) and any(is_cue_like(x) for x in pc):
                    return pc, "show.patternCue"
            # Some variants might use "cues" or "cue" list
            for key in ("cues", "cue", "patternCue"):
                v = doc.get(key)
                if isinstance(v, list) and v and all(isinstance(x, dict) for x in v) and any(is_cue_like(x) for x in v):
                    return v, key
        # Deep scan (bounded)
        stack = [(doc, "root")]
        visited = 0
        while stack and visited < 20000:
            visited += 1
            cur, p = stack.pop()
            if isinstance(cur, dict):
                for k, v in cur.items():
                    np = f"{p}.{k}"
                    if isinstance(v, list) and v and all(isinstance(x, dict) for x in v) and any(is_cue_like(x) for x in v):
                        return v, np
                    if isinstance(v, (dict, list)):
                        stack.append((v, np))
            elif isinstance(cur, list):
                for i, v in enumerate(cur[:200]):  # avoid insane lists
                    np = f"{p}[{i}]"
                    if isinstance(v, (dict, list)):
                        stack.append((v, np))
    except Exception:
        return None
    return None


def get_cue_label(cue: Dict[str, Any], idx: int) -> str:
    name = cue.get("name") or cue.get("title") or cue.get("label") or ""
    uuid = cue.get("uuid") or cue.get("id") or cue.get("cueId") or ""
    if name and uuid:
        return f"{idx+1:03d}  {name}   ({uuid})"
    if name:
        return f"{idx+1:03d}  {name}"
    if uuid:
        return f"{idx+1:03d}  ({uuid})"
    return f"{idx+1:03d}  <cue>"


def pretty_json(obj: Any) -> str:
    return json.dumps(obj, ensure_ascii=False, indent=2, sort_keys=False)


@dataclass
class Lang:
    data: Dict[str, str]

    def t(self, msg_key: str, **kwargs) -> str:
        s = self.data.get(msg_key, msg_key)
        try:
            return s.format(**kwargs)
        except Exception:
            return s


def load_lang(lang_code: str = "de") -> Lang:
    base_dir = Path(__file__).resolve().parent
    lang_path_json = base_dir / "lang" / f"{lang_code}.json"
    lang_path_cfg = base_dir / "lang" / f"{lang_code}.cfg"

    path = lang_path_json if lang_path_json.exists() else lang_path_cfg

    if not path.exists():
        lang_obj = Lang({"app_title": "Maestro Group Propagation Tool", "language": "Language:"})
        setattr(lang_obj, "code", lang_code)
        return lang_obj

    try:
        lang_obj = Lang(read_json(path))
        setattr(lang_obj, "code", lang_code)
        return lang_obj
    except Exception:
        lang_obj = Lang({"app_title": "Maestro Group Propagation Tool", "language": "Language:"})
        setattr(lang_obj, "code", lang_code)
        return lang_obj


class App(ttk.Frame):
    def __init__(self, master: tk.Tk, lang: Lang):
        super().__init__(master)
        self.master = master
        self.lang = lang

        self.source_path: Optional[Path] = None
        self.source_doc: Any = None
        self.source_cues: List[Dict[str, Any]] = []
        self.source_cues_path: str = ""
        self.selected_source_cue_index: Optional[int] = None

        self.target_folder: Optional[Path] = None
        self.target_files: List[Path] = []

        self.var_source_group = tk.IntVar(value=4)
        self.var_target_group = tk.IntVar(value=4)
        self.var_make_backup = tk.BooleanVar(value=True)

        self.var_folder_path = tk.StringVar(value=self.lang.t('no_folder_selected'))
        self.var_source_info = tk.StringVar(value=self.lang.t('no_source_loaded'))

        self.file_vars: List[tk.BooleanVar] = []

        self._build_ui()
        self._set_enabled(False)

    def _build_ui(self) -> None:
        self.master.title(self.lang.t("app_title"))
        self.pack(fill="both", expand=True)
        self.master.minsize(980, 640)

                # Language selector (DE/EN/FR/ES)
        self.var_lang = tk.StringVar(value=getattr(self.lang, "code", "de"))
        topbar = ttk.Frame(self)
        topbar.grid(row=0, column=0, columnspan=2, sticky="ew", padx=8, pady=(8, 0))
        self.lbl_lang = ttk.Label(topbar, text=self.lang.t("language"))
        self.lbl_lang.grid(row=0, column=0, sticky="w")
        self.cmb_lang = ttk.Combobox(topbar, textvariable=self.var_lang, values=["de", "en", "fr", "es"], width=5, state="readonly")
        self.cmb_lang.grid(row=0, column=1, sticky="w", padx=(6, 0))
        self.cmb_lang.bind("<<ComboboxSelected>>", self.on_language_changed)

        # Shift following frames down by +1 row (we will adjust grid rows below)

        # Layout: left selection / right preview / bottom log
        self.columnconfigure(0, weight=1, uniform="col")
        self.columnconfigure(1, weight=1, uniform="col")
        self.rowconfigure(3, weight=1)

        # --- Source frame
        self.lf_src = ttk.LabelFrame(self, text=self.lang.t("src_frame"))
        src = self.lf_src
        src.grid(row=1, column=0, sticky="nsew", padx=8, pady=8)
        src.columnconfigure(0, weight=1)
        src.rowconfigure(2, weight=1)

        self.btn_load_source = ttk.Button(src, text=self.lang.t("load_source_btn"), command=self.load_source)
        btn_load = self.btn_load_source
        btn_load.grid(row=0, column=0, sticky="ew", padx=6, pady=(6, 4))

        self.lbl_source = ttk.Label(src, textvariable=self.var_source_info, wraplength=440, justify='left')
        self.lbl_source.grid(row=1, column=0, sticky="ew", padx=6, pady=(0, 6))

        self.lst_cues = tk.Listbox(src, height=10, exportselection=False, bg='white', fg='black')
        self.lst_cues.grid(row=2, column=0, sticky="nsew", padx=6, pady=(0, 6))
        self.lst_cues.bind("<<ListboxSelect>>", self.on_source_cue_selected)

        grp_frame = ttk.Frame(src)
        grp_frame.grid(row=3, column=0, sticky="ew", padx=6, pady=(0, 6))
        ttk.Label(grp_frame, text=self.lang.t("source_group")).grid(row=0, column=0, sticky="w")
        for i in range(1, 5):
            ttk.Radiobutton(grp_frame, text=str(i), variable=self.var_source_group, value=i, command=self.refresh_preview).grid(
                row=0, column=i, padx=4, sticky="w"
            )

        # --- Preview frame
        self.lf_prev = ttk.LabelFrame(self, text=self.lang.t("preview_frame"))
        prev = self.lf_prev
        prev.grid(row=1, column=1, sticky="nsew", padx=8, pady=8)
        prev.columnconfigure(0, weight=1)
        prev.rowconfigure(0, weight=1)

        self.txt_preview = tk.Text(prev, height=22, wrap="none")
        self.txt_preview.grid(row=0, column=0, sticky="nsew", padx=6, pady=6)
        self.txt_preview.configure(font=("Courier New", 10))
        # scrollbars
        ysb = ttk.Scrollbar(prev, orient="vertical", command=self.txt_preview.yview)
        ysb.grid(row=0, column=1, sticky="ns", pady=6)
        xsb = ttk.Scrollbar(prev, orient="horizontal", command=self.txt_preview.xview)
        xsb.grid(row=1, column=0, sticky="ew", padx=6)
        self.txt_preview.configure(yscrollcommand=ysb.set, xscrollcommand=xsb.set)
        # --- Target selection frame
        self.lf_tgt = ttk.LabelFrame(self, text=self.lang.t("target_frame"))
        tgt = self.lf_tgt
        tgt.grid(row=2, column=0, columnspan=2, sticky="nsew", padx=8, pady=(0, 8))
        tgt.columnconfigure(0, weight=0)
        tgt.columnconfigure(1, weight=1)
        tgt.rowconfigure(2, weight=1)

        # Left control column
        left = ttk.Frame(tgt)
        left.grid(row=0, column=0, rowspan=3, sticky="nsw", padx=6, pady=6)

        ttk.Label(left, text=self.lang.t("target_group")).grid(row=0, column=0, sticky="w")
        # Radios vertical (1..4)
        for i in range(1, 5):
            ttk.Radiobutton(left, text=str(i), variable=self.var_target_group, value=i).grid(row=i, column=0, sticky="w", pady=1)

        # Backup in its own row under group selection
        ttk.Checkbutton(left, text=self.lang.t("make_backup"), variable=self.var_make_backup).grid(row=5, column=0, sticky="w", pady=(8, 0))

        ttk.Separator(left, orient="horizontal").grid(row=6, column=0, sticky="ew", pady=10)

        self.btn_all_on = ttk.Button(left, text=self.lang.t("select_all"), command=lambda: self.set_all_files(True))
        self.btn_all_on.grid(row=7, column=0, sticky="ew")
        self.btn_all_off = ttk.Button(left, text=self.lang.t("select_none"), command=lambda: self.set_all_files(False))
        self.btn_all_off.grid(row=8, column=0, sticky="ew", pady=(6, 0))

        self.btn_apply = ttk.Button(left, text=self.lang.t("apply_btn"), command=self.apply_changes)
        self.btn_apply.grid(row=9, column=0, sticky="ew", pady=(14, 0))

        # Right column: folder chooser + file list
        top_row = ttk.Frame(tgt)
        top_row.grid(row=0, column=1, sticky="ew", padx=6, pady=(6, 2))
        top_row.columnconfigure(0, weight=1)

        self.btn_folder = ttk.Button(top_row, text=self.lang.t("choose_folder_btn"), command=self.choose_folder)
        btn_folder = self.btn_folder
        btn_folder.grid(row=0, column=1, padx=(12, 0), sticky="e")

        self.ent_folder = ttk.Entry(tgt, textvariable=self.var_folder_path, state="readonly")
        self.ent_folder.grid(row=1, column=1, sticky="ew", padx=6, pady=(0, 6))

        # File list with checkboxes
        self.files_canvas = tk.Canvas(tgt, height=300)
        self.files_canvas.grid(row=2, column=1, sticky="nsew", padx=6, pady=(0, 6))
        self.files_scroll = ttk.Scrollbar(tgt, orient="vertical", command=self.files_canvas.yview)
        self.files_scroll.grid(row=2, column=2, sticky="ns", pady=(0, 6))
        self.files_canvas.configure(yscrollcommand=self.files_scroll.set)

        self.files_frame = ttk.Frame(self.files_canvas)
        self.files_canvas.create_window((0, 0), window=self.files_frame, anchor="nw")
        self.files_frame.bind("<Configure>", lambda e: self.files_canvas.configure(scrollregion=self.files_canvas.bbox("all")))

        # --- Log frame

        self.lf_log = ttk.LabelFrame(self, text=self.lang.t("log_frame"))
        logf = self.lf_log
        logf.grid(row=3, column=0, columnspan=2, sticky="nsew", padx=8, pady=(0, 8))
        logf.columnconfigure(0, weight=1)
        logf.rowconfigure(0, weight=1)

        self.txt_log = tk.Text(logf, height=6, wrap='word')
        self.txt_log.grid(row=0, column=0, sticky="nsew", padx=6, pady=6)
        self.txt_log.configure(font=("Segoe UI", 10))
        ysb2 = ttk.Scrollbar(logf, orient="vertical", command=self.txt_log.yview)
        ysb2.grid(row=0, column=1, sticky="ns", pady=6)
        self.txt_log.configure(yscrollcommand=ysb2.set)


        # --- Footer (author + help link)
        footer = ttk.Frame(self)
        footer.grid(row=4, column=0, columnspan=2, sticky="ew", padx=8, pady=(0, 8))
        footer.columnconfigure(0, weight=1)

        self.lbl_footer = ttk.Label(footer, text=self.lang.t("footer_author_duration"))
        self.lbl_footer.grid(row=0, column=0, sticky="w")

        self.lbl_help = ttk.Label(footer, text=self.lang.t("help_link_label"), cursor="hand2")
        self.lbl_help.grid(row=0, column=1, sticky="e")
        try:
            self.lbl_help.configure(foreground="#0000EE")
        except Exception:
            pass

        def _open_help(_evt=None):
            url = self.lang.t("help_link_url")
            try:
                webbrowser.open(url)
            except Exception:
                pass

        self.lbl_help.bind("<Button-1>", _open_help)



    def _set_enabled(self, enabled: bool) -> None:
        # Called after loading a source show.
        state = "normal" if enabled else "disabled"
        self.lst_cues.configure(state=state)
        # Radiobuttons are ttk - can't configure all easily; that's fine, preview will just show message.
        self.btn_all_on.configure(state=state)
        self.btn_all_off.configure(state=state)
        self.btn_apply.configure(state=state)

    def log(self, msg: str) -> None:
        ts = _dt.datetime.now().strftime("%H:%M:%S")
        self.txt_log.insert("end", f"[{ts}] {msg}\n")
        self.txt_log.see("end")

    def load_source(self) -> None:
        path = filedialog.askopenfilename(
            title=self.lang.t("load_source_title"),
            filetypes=[(self.lang.t("json_files"), "*.json"), (self.lang.t("all_files"), "*.*")]
        )
        if not path:
            return
        try:
            p = Path(path)
            doc = read_json(p)
            found = find_cue_list(doc)
            if not found:
                messagebox.showerror(self.lang.t("error"), self.lang.t("no_cues_found"))
                return
            cues, cue_path = found
            self.source_path = p
            self.source_doc = doc
            self.source_cues = cues
            self.source_cues_path = cue_path

            self.var_source_info.set(self.lang.t('source_loaded', path=str(p), cuepath=cue_path, n=len(cues)))

            # Ensure listbox is enabled while (re)populating items (Linux themes can otherwise render empty until next load)
            try:
                self.lst_cues.configure(state="normal")
            except Exception:
                pass
            self.lst_cues.delete(0, "end")
            for i, cue in enumerate(cues):
                self.lst_cues.insert("end", get_cue_label(cue, i))

            # select first by default
            if cues:
                self.lst_cues.selection_clear(0, "end")
                self.lst_cues.selection_set(0)
                self.lst_cues.activate(0)
                self.selected_source_cue_index = 0
                self.refresh_preview()
                try:
                    self.master.update_idletasks()
                except Exception:
                    pass

            self._set_enabled(True)
            self.log(self.lang.t("log_loaded_source", n=len(cues), path=str(p)))
        except Exception as e:
            traceback.print_exc()
            messagebox.showerror(self.lang.t("error"), f"{self.lang.t('failed_load')}\n\n{e}")

    def on_source_cue_selected(self, _evt=None) -> None:
        try:
            sel = self.lst_cues.curselection()
            if not sel:
                self.selected_source_cue_index = None
                self.refresh_preview()
                return
            self.selected_source_cue_index = int(sel[0])
            self.refresh_preview()
        except Exception:
            self.selected_source_cue_index = None
            self.refresh_preview()


    def on_language_changed(self, _evt=None) -> None:
        lang_code = (self.var_lang.get() or "de").strip().lower()
        try:
            self.lang = load_lang(lang_code)
        except Exception:
            self.lang = load_lang("de")
            self.var_lang.set("de")
        # Update window title and all UI labels/buttons
        self._apply_language_to_ui()
        # Refresh preview and folder labels
        self.refresh_preview()

    def _apply_language_to_ui(self) -> None:
        # Window title
        try:
            self.master.title(self.lang.t("app_title"))
        except Exception:
            pass
        # LabelFrame texts and common labels/buttons
        try:
            self.lf_src.configure(text=self.lang.t("src_frame"))
            self.btn_load_source.configure(text=self.lang.t("load_source_btn"))
            if self.source_path:
                self.lbl_source.configure(text=self.lang.t("source_loaded", path=str(self.source_path), cuepath=self.source_cues_path, n=len(self.source_cues)))
            else:
                self.lbl_source.configure(text=self.lang.t("no_source_loaded"))

            self.lf_prev.configure(text=self.lang.t("preview_frame"))
            self.lf_tgt.configure(text=self.lang.t("target_frame"))
            self.lf_log.configure(text=self.lang.t("log_frame"))

            self.lbl_folder.configure(text=self.lang.t("folder_selected", path=str(self.target_folder)) if self.target_folder else self.lang.t("no_folder_selected"))

            self.chk_backup.configure(text=self.lang.t("make_backup"))
            self.btn_folder.configure(text=self.lang.t("choose_folder_btn"))
            self.btn_all_on.configure(text=self.lang.t("select_all"))
            self.btn_all_off.configure(text=self.lang.t("select_none"))
            self.btn_apply.configure(text=self.lang.t("apply_btn"))

            self.lbl_lang.configure(text=self.lang.t("language"))

            # Footer
            try:
                self.lbl_footer.configure(text=self.lang.t("footer_author_duration"))
                self.lbl_help.configure(text=self.lang.t("help_link_label"))
            except Exception:
                pass
        except Exception:
            # Do not crash UI on partial updates
            pass

    def refresh_preview(self) -> None:
        self.txt_preview.delete("1.0", "end")
        if self.selected_source_cue_index is None or not self.source_cues:
            self.txt_preview.insert("end", self.lang.t("preview_no_selection"))
            return
        sg = int(self.var_source_group.get())
        key = GROUP_KEY_BY_NUM.get(sg)
        if not key:
            self.txt_preview.insert("end", self.lang.t("preview_no_group"))
            return
        cue = self.source_cues[self.selected_source_cue_index]
        block = cue.get(key)
        if block is None:
            self.txt_preview.insert("end", self.lang.t("preview_group_missing", g=sg, group_key=key))
            return
        header = self.lang.t("preview_header", cue=get_cue_label(cue, self.selected_source_cue_index), g=sg, group_key=key)
        self.txt_preview.insert("end", header + "\n\n")
        self.txt_preview.insert("end", pretty_json(block))

    def choose_folder(self) -> None:
        folder = filedialog.askdirectory(title=self.lang.t("choose_folder_title"))
        if not folder:
            return
        self.target_folder = Path(folder)
        self.var_folder_path.set(self.lang.t('folder_selected', path=str(self.target_folder)))
        self._load_target_files()

    def _load_target_files(self) -> None:
        # Clear previous
        for w in self.files_frame.winfo_children():
            w.destroy()
        self.file_vars.clear()
        self.target_files.clear()

        if not self.target_folder:
            return

        # List only *.json files, excluding language files if inside same tree
        files = sorted([p for p in self.target_folder.glob("*.json") if p.is_file()])
        self.target_files = files

        if not files:
            ttk.Label(self.files_frame, text=self.lang.t("no_json_found")).grid(row=0, column=0, sticky="w")
            self.log(self.lang.t("log_no_json", path=str(self.target_folder)))
            return

        for i, p in enumerate(files):
            var = tk.BooleanVar(value=False)
            self.file_vars.append(var)
            cb = ttk.Checkbutton(self.files_frame, text=p.name, variable=var)
            cb.grid(row=i, column=0, sticky="w", padx=2, pady=2)

        self.log(self.lang.t("log_found_files", n=len(files), path=str(self.target_folder)))

    def set_all_files(self, value: bool) -> None:
        for v in self.file_vars:
            v.set(value)

    def _get_selected_target_files(self) -> List[Path]:
        out: List[Path] = []
        for p, v in zip(self.target_files, self.file_vars):
            if v.get():
                out.append(p)
        return out

    def apply_changes(self) -> None:
        if self.source_doc is None or self.source_path is None or not self.source_cues:
            messagebox.showerror(self.lang.t("error"), self.lang.t("no_source_loaded"))
            return
        if self.selected_source_cue_index is None:
            messagebox.showerror(self.lang.t("error"), self.lang.t("no_cue_selected"))
            return
        if not self.target_folder:
            messagebox.showerror(self.lang.t("error"), self.lang.t("no_folder"))
            return

        selected_files = self._get_selected_target_files()
        if not selected_files:
            messagebox.showerror(self.lang.t("error"), self.lang.t("no_targets_selected"))
            return

        sg = int(self.var_source_group.get())
        tg = int(self.var_target_group.get())
        s_key = GROUP_KEY_BY_NUM[sg]
        t_key = GROUP_KEY_BY_NUM[tg]

        source_cue = self.source_cues[self.selected_source_cue_index]
        source_block = source_cue.get(s_key)
        if source_block is None:
            messagebox.showerror(self.lang.t("error"), self.lang.t("source_group_missing", g=sg))
            return

        confirm = messagebox.askyesno(
            self.lang.t("confirm_title"),
            self.lang.t(
                "confirm_text",
                n=len(selected_files),
                sg=sg,
                tg=tg,
                sk=s_key,
                tk=t_key,
            )
        )
        if not confirm:
            return

        # Apply to each file
        total_files_ok = 0
        total_files_skipped = 0
        total_cues_modified = 0

        for fp in selected_files:
            try:
                doc = read_json(fp)
                found = find_cue_list(doc)
                if not found:
                    self.log(self.lang.t("log_skip_no_cues", file=fp.name))
                    total_files_skipped += 1
                    continue

                cues, cue_path = found
                modified_in_file = 0

                for cue in cues:
                    # Ensure dict
                    if not isinstance(cue, dict):
                        continue
                    cue[t_key] = copy.deepcopy(source_block)
                    modified_in_file += 1

                if modified_in_file == 0:
                    self.log(self.lang.t("log_skip_no_mod", file=fp.name))
                    total_files_skipped += 1
                    continue

                if self.var_make_backup.get():
                    backup = fp.with_suffix(fp.suffix + f".bak.{now_ts()}")
                    try:
                        backup.write_bytes(fp.read_bytes())
                    except Exception as e:
                        self.log(self.lang.t("log_backup_failed", file=fp.name, err=str(e)))
                        # continue anyway, but warn
                write_json(fp, doc)

                self.log(self.lang.t("log_modified_file", file=fp.name, cues=modified_in_file, cuepath=cue_path))
                total_files_ok += 1
                total_cues_modified += modified_in_file

            except Exception as e:
                self.log(self.lang.t("log_error_file", file=fp.name, err=str(e)))
                total_files_skipped += 1

        messagebox.showinfo(
            self.lang.t("done_title"),
            self.lang.t("done_text", ok=total_files_ok, skip=total_files_skipped, cues=total_cues_modified),
        )


def main() -> int:
    lang = load_lang("de")
    root = tk.Tk()
    try:
        # Use a modern theme where available
        style = ttk.Style()
        if "clam" in style.theme_names():
            style.theme_use("clam")
    except Exception:
        pass

    app = App(root, lang)
    # Make mouse wheel work on file list scrolling
    def _on_mousewheel(event):
        try:
            app.files_canvas.yview_scroll(int(-1*(event.delta/120)), "units")
        except Exception:
            pass
    app.files_canvas.bind_all("<MouseWheel>", _on_mousewheel)

    root.mainloop()
    return 0


if __name__ == "__main__":
    raise SystemExit(main())
