#!/usr/bin/env python3
from __future__ import annotations

import configparser
import json
import os
import shutil
import subprocess
import threading
import time
import webbrowser
from dataclasses import dataclass
from datetime import datetime, timedelta
from pathlib import Path
import tkinter as tk
from tkinter import ttk, messagebox

APP_TITLE = "Photobooth Export Tool"
SOURCE_BASE = Path("/var/www/html/data")
EVENT_CUTOFF_HOUR = 10
EXPORT_PREFIX = "Fotobox vom "
CONFIG_PATH = Path(__file__).resolve().parent / "usb-sync.cfg"
LANG_DIR = Path(__file__).resolve().parent / "lang"
SUPPORTED_LANGS = ["de", "en", "es", "fr"]
DEFAULT_LANG = "de"
DEFAULT_LANGUAGE_NAMES = {
    "de": "Deutsch",
    "en": "English",
    "es": "Español",
    "fr": "Français",
}


@dataclass
class UsbTarget:
    device: str
    name: str
    label: str
    fstype: str
    mountpoint: Path
    free_bytes: int
    size_label: str


@dataclass
class DirStats:
    path: Path
    file_count: int
    total_bytes: int


def human_size(num_bytes: int) -> str:
    units = ["B", "KB", "MB", "GB", "TB"]
    value = float(num_bytes)
    for unit in units:
        if value < 1024.0 or unit == units[-1]:
            return f"{value:.1f} {unit}"
        value /= 1024.0
    return f"{num_bytes} B"


def run_command(cmd: list[str], check: bool = False) -> subprocess.CompletedProcess:
    return subprocess.run(cmd, capture_output=True, text=True, check=check)


def lsblk_json() -> dict:
    result = run_command([
        "lsblk", "-J",
        "-o", "NAME,LABEL,FSTYPE,MOUNTPOINT,RM,TRAN,TYPE,SIZE",
    ], check=True)
    return json.loads(result.stdout)


def find_first_usb_target() -> UsbTarget | None:
    try:
        data = lsblk_json()
    except Exception:
        return None

    candidates: list[UsbTarget] = []

    def visit(entry: dict, inherited_usb: bool = False):
        tran = (entry.get("tran") or "").lower()
        rm = str(entry.get("rm") or "0")
        is_usbish = inherited_usb or tran == "usb" or rm == "1"

        if entry.get("type") == "part":
            mountpoint = entry.get("mountpoint")
            if mountpoint and is_usbish:
                mount = Path(mountpoint)
                if mount.exists() and os.access(mount, os.W_OK):
                    try:
                        usage = shutil.disk_usage(mount)
                    except Exception:
                        usage = None
                    candidates.append(
                        UsbTarget(
                            device=f"/dev/{entry.get('name')}",
                            name=entry.get("name") or "",
                            label=entry.get("label") or "",
                            fstype=entry.get("fstype") or "",
                            mountpoint=mount,
                            free_bytes=usage.free if usage else 0,
                            size_label=entry.get("size") or "",
                        )
                    )

        for child in entry.get("children", []) or []:
            visit(child, inherited_usb=is_usbish)

    for block in data.get("blockdevices", []):
        visit(block, inherited_usb=False)

    return candidates[0] if candidates else None


def scan_source_dirs(base: Path) -> list[DirStats]:
    if not base.exists():
        return []
    dirs: list[DirStats] = []
    for child in sorted(base.iterdir(), key=lambda p: p.name.lower()):
        if not child.is_dir():
            continue
        file_count = 0
        total_bytes = 0
        for root, _, files in os.walk(child):
            for filename in files:
                fp = Path(root) / filename
                try:
                    st = fp.stat()
                except OSError:
                    continue
                file_count += 1
                total_bytes += st.st_size
        dirs.append(DirStats(path=child, file_count=file_count, total_bytes=total_bytes))
    return dirs


def event_date_for_timestamp(ts: float):
    dt = datetime.fromtimestamp(ts)
    if dt.hour < EVENT_CUTOFF_HOUR:
        dt = dt - timedelta(days=1)
    return dt.date()


def unique_event_folder(base_mount: Path, event_date) -> Path:
    base_name = f"{EXPORT_PREFIX}{event_date.strftime('%d.%m.%Y')}"
    candidate = base_mount / base_name
    if not candidate.exists():
        return candidate
    index = 1
    while True:
        candidate = base_mount / f"{base_name} ({index})"
        if not candidate.exists():
            return candidate
        index += 1


class ExportApp(tk.Tk):
    def __init__(self):
        super().__init__()
        self.title(APP_TITLE)
        self.geometry("1040x780")
        self.minsize(940, 700)

        self.usb_target: UsbTarget | None = None
        self.dir_stats: list[DirStats] = []
        self.dir_vars: dict[Path, tk.BooleanVar] = {}
        self.event_folder_map: dict[str, Path] = {}
        self.export_session_log: list[str] = []
        self.copy_thread: threading.Thread | None = None
        self.copy_finished_ok = False
        self.unmounted_ok = False

        self.total_bytes = 0
        self.copied_bytes = 0
        self.total_files = 0
        self.copied_files = 0

        self.translations: dict[str, str] = {}
        self.cfg = configparser.ConfigParser()
        self.saved_selected_dir_names: set[str] = set()
        self.language_code = DEFAULT_LANG
        self.language_var = tk.StringVar(value=DEFAULT_LANG)
        self._load_config()
        self._load_language(self.language_code)

        self.usb_frame = None
        self.src_frame = None
        self.prog_frame = None
        self.log_frame = None
        self.refresh_btn = None
        self.start_btn = None
        self.eject_btn = None
        self.shutdown_btn = None
        self.language_label = None
        self.footer_label = None

        self._build_ui()
        self.refresh_state(initial=True)

    def tr(self, key: str, **kwargs) -> str:
        text = self.translations.get(key, key)
        if kwargs:
            try:
                return text.format(**kwargs)
            except Exception:
                return text
        return text

    def _load_config(self):
        if CONFIG_PATH.exists():
            self.cfg.read(CONFIG_PATH, encoding="utf-8")
        if "general" not in self.cfg:
            self.cfg["general"] = {}
        if "selection" not in self.cfg:
            self.cfg["selection"] = {}

        lang = self.cfg["general"].get("language", DEFAULT_LANG).strip().lower()
        if lang not in SUPPORTED_LANGS:
            lang = DEFAULT_LANG
        self.language_code = lang

        saved = self.cfg["selection"].get("dirs", "").strip()
        self.saved_selected_dir_names = {x.strip() for x in saved.split(",") if x.strip()}

    def _save_config(self):
        self.cfg["general"]["language"] = self.language_code
        selected_names = [path.name for path, var in self.dir_vars.items() if var.get()]
        self.cfg["selection"]["dirs"] = ",".join(selected_names)
        with CONFIG_PATH.open("w", encoding="utf-8") as f:
            self.cfg.write(f)

    def _load_language(self, code: str):
        lang_file = LANG_DIR / f"{code}.json"
        try:
            self.translations = json.loads(lang_file.read_text(encoding="utf-8"))
        except Exception:
            fallback = LANG_DIR / f"{DEFAULT_LANG}.json"
            self.translations = json.loads(fallback.read_text(encoding="utf-8"))
            self.language_code = DEFAULT_LANG
        self.language_var.set(self.language_code)

    def _build_ui(self):
        pad = {"padx": 10, "pady": 8}

        top_frame = ttk.Frame(self)
        top_frame.pack(fill="x", **pad)
        self.language_label = ttk.Label(top_frame, text="")
        self.language_label.pack(side="left")

        language_values = [f"{code} - {DEFAULT_LANGUAGE_NAMES[code]}" for code in SUPPORTED_LANGS]
        self.language_combo = ttk.Combobox(top_frame, values=language_values, state="readonly", width=18)
        self.language_combo.pack(side="left", padx=(8, 10))
        self.language_combo.bind("<<ComboboxSelected>>", self.on_language_selected)

        self.refresh_btn = ttk.Button(top_frame, command=self.refresh_state)
        self.refresh_btn.pack(side="left")

        self.usb_frame = ttk.LabelFrame(self, text="")
        self.usb_frame.pack(fill="x", **pad)
        self.usb_info_var = tk.StringVar(value="")
        self.usb_label = ttk.Label(self.usb_frame, textvariable=self.usb_info_var, justify="left")
        self.usb_label.pack(anchor="w", padx=10, pady=10)

        self.src_frame = ttk.LabelFrame(self, text="")
        self.src_frame.pack(fill="x", **pad)
        self.src_inner = ttk.Frame(self.src_frame)
        self.src_inner.pack(fill="x", padx=10, pady=10)

        self.prog_frame = ttk.LabelFrame(self, text="")
        self.prog_frame.pack(fill="x", **pad)
        self.progress_var = tk.DoubleVar(value=0.0)
        self.progress = ttk.Progressbar(self.prog_frame, orient="horizontal", mode="determinate",
                                        variable=self.progress_var, maximum=100)
        self.progress.pack(fill="x", padx=10, pady=(10, 6))
        self.progress_text_var = tk.StringVar(value="")
        ttk.Label(self.prog_frame, textvariable=self.progress_text_var).pack(anchor="w", padx=10, pady=(0, 10))

        action_frame = ttk.Frame(self)
        action_frame.pack(fill="x", **pad)
        self.start_btn = ttk.Button(action_frame, command=self.start_copy, state="disabled")
        self.start_btn.pack(side="left", padx=(0, 10))
        self.eject_btn = ttk.Button(action_frame, command=self.eject_usb, state="disabled")
        self.eject_btn.pack(side="left", padx=(0, 10))
        self.shutdown_btn = ttk.Button(action_frame, command=self.shutdown_pi, state="disabled")
        self.shutdown_btn.pack(side="left", padx=(0, 16))

        self.footer_prefix_var = tk.StringVar(value="")
        self.footer_prefix_label = ttk.Label(action_frame, textvariable=self.footer_prefix_var)
        self.footer_prefix_label.pack(side="right")
        self.footer_label = tk.Label(
            action_frame,
            text="liberty-dj-tools.de",
            fg="#0645AD",
            cursor="hand2",
            font=("TkDefaultFont", 9, "underline"),
        )
        self.footer_label.pack(side="right", padx=(0, 4))
        self.footer_label.bind("<Button-1>", lambda _e: webbrowser.open("https://liberty-dj-tools.de"))

        self.log_frame = ttk.LabelFrame(self, text="")
        self.log_frame.pack(fill="both", expand=True, **pad)
        self.log_text = tk.Text(self.log_frame, height=18, wrap="word")
        self.log_text.pack(side="left", fill="both", expand=True, padx=(10, 0), pady=10)
        scroll = ttk.Scrollbar(self.log_frame, orient="vertical", command=self.log_text.yview)
        scroll.pack(side="right", fill="y", padx=(0, 10), pady=10)
        self.log_text.configure(yscrollcommand=scroll.set)

        self._apply_translations()

    def _apply_translations(self):
        self.language_label.config(text=self.tr("language_label"))
        self.refresh_btn.config(text=self.tr("refresh_button"))
        self.usb_frame.config(text=self.tr("usb_target_frame"))
        self.src_frame.config(text=self.tr("source_dirs_frame", source_base=str(SOURCE_BASE)))
        self.prog_frame.config(text=self.tr("copy_process_frame"))
        self.log_frame.config(text=self.tr("log_frame"))
        self.start_btn.config(text=self.tr("start_button"))
        self.eject_btn.config(text=self.tr("eject_button"))
        self.shutdown_btn.config(text=self.tr("shutdown_button"))
        self.footer_prefix_var.set(self.tr("footer_prefix"))
        self.title(self.tr("app_title"))
        self._set_language_combo_selection()
        self._update_usb_info_text()
        if not self.copy_thread:
            if self.progress_var.get() == 0:
                self.progress_text_var.set(self.tr("ready_status"))
        self._rebuild_source_list()

    def _set_language_combo_selection(self):
        display = f"{self.language_code} - {DEFAULT_LANGUAGE_NAMES[self.language_code]}"
        self.language_combo.set(display)

    def on_language_selected(self, _event=None):
        raw = self.language_combo.get().split("-")[0].strip().lower()
        if raw not in SUPPORTED_LANGS:
            return
        self.language_code = raw
        self._load_language(raw)
        self._save_config()
        self._apply_translations()

    def log(self, message: str):
        timestamp = datetime.now().strftime("%H:%M:%S")
        line = f"[{timestamp}] {message}"
        self.export_session_log.append(line)
        self.log_text.insert("end", line + "\n")
        self.log_text.see("end")
        self.update_idletasks()

    def clear_log(self):
        self.log_text.delete("1.0", "end")
        self.export_session_log.clear()

    def refresh_state(self, initial: bool = False):
        self.copy_finished_ok = False
        self.unmounted_ok = False
        self.eject_btn.config(state="disabled")
        self.shutdown_btn.config(state="disabled")
        self.usb_target = find_first_usb_target()
        self.dir_stats = scan_source_dirs(SOURCE_BASE)
        self._update_usb_info_text()
        self._rebuild_source_list(initial=initial)
        self._refresh_button_states()
        self.update_idletasks()

    def _update_usb_info_text(self):
        if self.usb_target:
            label = self.usb_target.label or self.tr("no_label")
            self.usb_info_var.set(
                self.tr(
                    "usb_info_found",
                    device=self.usb_target.device,
                    label=label,
                    fstype=self.usb_target.fstype or self.tr("unknown_fs"),
                    mountpoint=self.usb_target.mountpoint,
                    free=human_size(self.usb_target.free_bytes),
                )
            )
        else:
            self.usb_info_var.set(self.tr("usb_not_found"))

    def _default_dir_selected(self, ds: DirStats, initial: bool) -> bool:
        if initial and self.saved_selected_dir_names:
            return ds.path.name in self.saved_selected_dir_names
        return True

    def _on_selection_changed(self):
        self._save_config()
        self._refresh_button_states()

    def _rebuild_source_list(self, initial: bool = False):
        for widget in self.src_inner.winfo_children():
            widget.destroy()

        self.dir_vars = {}
        if not self.dir_stats:
            ttk.Label(self.src_inner, text=self.tr("no_subfolders_found")).pack(anchor="w")
            return

        header = ttk.Frame(self.src_inner)
        header.pack(fill="x", pady=(0, 5))
        ttk.Label(header, text=self.tr("col_select"), width=12).pack(side="left")
        ttk.Label(header, text=self.tr("col_folder"), width=24).pack(side="left")
        ttk.Label(header, text=self.tr("col_files"), width=12).pack(side="left")
        ttk.Label(header, text=self.tr("col_size"), width=16).pack(side="left")

        for ds in self.dir_stats:
            row = ttk.Frame(self.src_inner)
            row.pack(fill="x", pady=2)
            var = tk.BooleanVar(value=self._default_dir_selected(ds, initial))
            self.dir_vars[ds.path] = var
            cb = ttk.Checkbutton(row, variable=var, command=self._on_selection_changed)
            cb.pack(side="left", padx=(0, 18))
            ttk.Label(row, text=ds.path.name, width=24).pack(side="left")
            ttk.Label(row, text=str(ds.file_count), width=12).pack(side="left")
            ttk.Label(row, text=human_size(ds.total_bytes), width=16).pack(side="left")

        self._save_config()

    def _refresh_button_states(self):
        any_checked = any(v.get() for v in self.dir_vars.values())
        can_start = bool(self.usb_target and self.dir_stats and any_checked and self.copy_thread is None)
        self.start_btn.config(state="normal" if can_start else "disabled")
        self.eject_btn.config(state="normal" if self.copy_finished_ok else "disabled")
        self.shutdown_btn.config(state="normal" if self.unmounted_ok else "disabled")
        self.update_idletasks()

    def selected_paths(self) -> list[Path]:
        return [path for path, var in self.dir_vars.items() if var.get()]

    def estimate_selected_bytes_and_files(self) -> tuple[int, int]:
        chosen = set(self.selected_paths())
        total_bytes = 0
        total_files = 0
        for ds in self.dir_stats:
            if ds.path in chosen:
                total_bytes += ds.total_bytes
                total_files += ds.file_count
        return total_bytes, total_files

    def start_copy(self):
        if self.copy_thread is not None:
            return
        if not self.usb_target:
            messagebox.showerror(self.tr("app_title"), self.tr("msg_no_usb"))
            return
        selected = self.selected_paths()
        if not selected:
            messagebox.showwarning(self.tr("app_title"), self.tr("msg_select_one"))
            return

        self.clear_log()
        self.copy_finished_ok = False
        self.unmounted_ok = False
        self.progress_var.set(0.0)
        self.copied_bytes = 0
        self.copied_files = 0
        self.total_bytes, self.total_files = self.estimate_selected_bytes_and_files()
        self.progress_text_var.set(self.tr("copy_starting_status"))
        self.event_folder_map = {}

        self.copy_thread = threading.Thread(target=self._copy_worker, daemon=True)
        self.copy_thread.start()
        self._poll_copy_thread()
        self._refresh_button_states()

    def _poll_copy_thread(self):
        if self.copy_thread and self.copy_thread.is_alive():
            self.after(250, self._poll_copy_thread)
            return
        self.copy_thread = None
        self._refresh_button_states()

    def _copy_worker(self):
        start_time = time.time()
        selected = self.selected_paths()
        self.log(self.tr("log_start_export", count=len(selected)))
        self.log(self.tr("log_usb_target", mountpoint=self.usb_target.mountpoint))

        errors = 0

        for source_dir in selected:
            self.log(self.tr("log_process_folder", folder=source_dir.name))
            for root, _, files in os.walk(source_dir):
                for filename in files:
                    src = Path(root) / filename
                    try:
                        st = src.stat()
                    except OSError as exc:
                        errors += 1
                        self.log(self.tr("log_error_stat", src=src, error=exc))
                        continue

                    event_date = event_date_for_timestamp(st.st_mtime)
                    event_key = event_date.isoformat()

                    if event_key not in self.event_folder_map:
                        event_root = unique_event_folder(self.usb_target.mountpoint, event_date)
                        event_root.mkdir(parents=True, exist_ok=False)
                        self.event_folder_map[event_key] = event_root
                        self.log(self.tr("log_created_target", folder=event_root.name))

                    event_root = self.event_folder_map[event_key]
                    rel_inside_source = src.relative_to(source_dir)
                    dest_dir = event_root / source_dir.name / rel_inside_source.parent
                    dest_dir.mkdir(parents=True, exist_ok=True)
                    dest = dest_dir / src.name

                    try:
                        self._copy_file_with_progress(src, dest)
                        shutil.copystat(src, dest, follow_symlinks=True)
                        self.copied_files += 1
                        self.log(self.tr("log_copied", src=src, dest=dest))
                    except Exception as exc:
                        errors += 1
                        self.log(self.tr("log_error_copy", src=src, error=exc))

        duration = time.time() - start_time
        summary = self.tr(
            "summary_export_done",
            copied_files=self.copied_files,
            total_files=self.total_files,
            copied_size=human_size(self.copied_bytes),
            total_size=human_size(self.total_bytes),
            duration=f"{duration:.1f}",
            errors=errors,
        )
        self.log(summary)
        self._write_log_to_usb(summary)
        self.progress_text_var.set(summary)
        self.copy_finished_ok = (errors == 0)

        if not self.copy_finished_ok:
            self.log(self.tr("log_eject_locked"))
        self._refresh_button_states()

    def _copy_file_with_progress(self, src: Path, dest: Path, chunk_size: int = 1024 * 1024):
        with open(src, "rb") as fsrc, open(dest, "wb") as fdst:
            while True:
                chunk = fsrc.read(chunk_size)
                if not chunk:
                    break
                fdst.write(chunk)
                self.copied_bytes += len(chunk)
                pct = 0.0 if self.total_bytes <= 0 else (self.copied_bytes / self.total_bytes) * 100.0
                self.progress_var.set(min(pct, 100.0))
                self.progress_text_var.set(
                    self.tr(
                        "copy_progress_status",
                        copied_files=self.copied_files,
                        total_files=self.total_files,
                        copied_size=human_size(self.copied_bytes),
                        total_size=human_size(self.total_bytes),
                    )
                )
                self.update_idletasks()

    def _write_log_to_usb(self, summary: str):
        if not self.usb_target:
            return
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        log_path = self.usb_target.mountpoint / f"photobooth_export_log_{timestamp}.txt"
        try:
            content = "\n".join(self.export_session_log) + "\n\n" + summary + "\n"
            log_path.write_text(content, encoding="utf-8")
            self.log(self.tr("log_saved", name=log_path.name))
        except Exception as exc:
            self.log(self.tr("log_error_save_log", error=exc))

    def eject_usb(self):
        if not self.usb_target:
            messagebox.showerror(self.tr("app_title"), self.tr("msg_no_usb_target"))
            return

        device = self.usb_target.device
        mountpoint = str(self.usb_target.mountpoint)
        self.log(self.tr("log_try_eject", device=device))

        commands = [
            ["udisksctl", "unmount", "-b", device],
            ["umount", device],
            ["umount", mountpoint],
            ["sudo", "umount", device],
            ["sudo", "umount", mountpoint],
        ]

        success = False
        last_error = ""
        for cmd in commands:
            try:
                result = run_command(cmd, check=False)
                if result.returncode == 0:
                    success = True
                    break
                last_error = (result.stderr or result.stdout or "").strip()
            except Exception as exc:
                last_error = str(exc)

        if success:
            self.log(self.tr("log_eject_success"))
            self.unmounted_ok = True
            self.copy_finished_ok = False
            self.usb_target = None
            self.usb_info_var.set(self.tr("usb_ejected"))
            self._refresh_button_states()
            self.update_idletasks()
        else:
            self.log(self.tr("log_eject_failed", error=last_error))
            messagebox.showerror(self.tr("app_title"), self.tr("msg_eject_failed", error=last_error))

    def shutdown_pi(self):
        if not self.unmounted_ok:
            messagebox.showwarning(self.tr("app_title"), self.tr("msg_shutdown_after_eject"))
            return

        if not messagebox.askyesno(self.tr("app_title"), self.tr("msg_shutdown_confirm")):
            return

        self.log(self.tr("log_shutdown_start"))
        commands = [
            ["shutdown", "-h", "now"],
            ["sudo", "shutdown", "-h", "now"],
        ]
        last_error = ""
        for cmd in commands:
            try:
                result = run_command(cmd, check=False)
                if result.returncode == 0:
                    return
                last_error = (result.stderr or result.stdout or "").strip()
            except Exception as exc:
                last_error = str(exc)

        self.log(self.tr("log_shutdown_failed", error=last_error))
        messagebox.showerror(self.tr("app_title"), self.tr("msg_shutdown_failed", error=last_error))


if __name__ == "__main__":
    app = ExportApp()
    app.mainloop()
